<?php declare(strict_types=1);
/**
 * JTL Plugin
 * Copyright (C) 2011 - 2020 Andreas Petermann
 *
 * PHP version 7.2+
 *
 * @copyright css-umsetzung
 * @author Andreas Petermann <info@css-umsetzung.de>
 * @package css_minValue
 * @license GNU
 * @version 1.00
 *
 */

namespace Plugin\css_minValue;

use JTL\Backend\Notification;
use JTL\Backend\NotificationEntry;
use JTL\Events\Dispatcher;
use JTL\Plugin\Bootstrapper;
use JTL\Shop;
use JTL\Router\Route;
use JTL\Minify\MinifyService;

class Bootstrap extends Bootstrapper {

	private $cssPlugin;
	private $_BootstrapVersion = '2.4';

	/**
	 * @inheritdoc
	 */
	public function boot(Dispatcher $dispatcher) {
		parent::boot($dispatcher);
		$plugin = $this->getPlugin();

		$dispatcher->listen('backend.notification', [$this, 'checkNotification']);

		if (Shop::isFrontend() === true || (isset($_REQUEST['kPlugin']) && $plugin->getID() == (int)$_REQUEST['kPlugin'])) {

			$this->cssPlugin = src\cssPlugin\cssPlugin::getInstance(
				$plugin,
				$this->getDB(),
				$this->getCache()
			);
		}
        if (Shop::isFrontend() === true) {
            $cfg = $this->cssPlugin->getConfig();
            if($cfg->pluginActive=='N') return;

            $dispatcher->listen('shop.hook.' . \HOOK_SMARTY_OUTPUTFILTER, function () use ($plugin) {
                $this->cssPlugin->hookSmartyOutputfilter();
            });


            $dispatcher->listen([
							'shop.hook.' . \HOOK_BESTELLVORGANG_PAGE,
	            'amazonpay.hook.' . 'afterGetPossibleShippingMethods'
	            ], function () use ($plugin) {
                $this->cssPlugin->hookBestellvorgangPage();
            });

        }  else {
	        if(isset($_REQUEST['io']) && $_REQUEST['io'] == $plugin->getPluginID()) {
		        $this->cssPlugin = src\cssPlugin\cssPlugin::getInstance(
			        $plugin,
			        $this->getDB(),
			        $this->getCache()
		        );
		        $dispatcher->listen('shop.hook.' . \HOOK_IO_HANDLE_REQUEST_ADMIN, function (array $args) use ($plugin)  {
			        $this->cssPlugin->hookIoHandleAdminRequest($args);
		        });
	        }
        }
    }

	public function columnExist($table, $field) {
		// false bedeutet das es die Spalte nicht gibt
		if ($this->tableExist($table)) {
			return $this->getDB()->query("Show columns from `" . $table . "` like '" . $field . "' ", 1);
		}
		return false;
	}

	public function tableExist($table) {
		// in Rows steht die Anzahl der Zeilen, false bedeutet das es die Tabelle nicht gibt
		$obj = $this->getDB()->query("SHOW TABLE STATUS  like '" . $table . "'", 1);
		if ($obj && (int)$obj->Rows == 0) {
			$o = $this->getDB()->query("SELECT COUNT(*) AS anzahl FROM " . $table . " WHERE 1", 1);
			$obj->Rows = (int)$o->anzahl;
			return $obj;
		}
		return $obj;
	}


	public function installed() {
		$plugin = $this->getPlugin();
		$obj = new \stdClass();
		$obj->cAktiv = 'Y';
		$obj->nFehlerhaft = '0';
		$this->getDB()->update('temailvorlage', 'kPlugin', $plugin->getID(), $obj);

		$minify = new MinifyService();
		$minify->flushCache();

		Shop::Container()->getCache()->flushTags(array(CACHING_GROUP_PLUGIN, CACHING_GROUP_TEMPLATE, 'cache_' . $plugin->getPluginID()));
		parent::installed();
	}

	public function updated($oldVersion, $newVersion) {
		$plugin = $this->getPlugin();


		if ($this->columnExist('css_licence_log', 'notifySubscriptionMail')) {
			$obj = new \stdClass();
			$obj->notifyUpdate = '';
			$obj->notifyUpdateMail = '';
			$this->getDB()->update('css_licence_log', 'cPluginID', $plugin->getPluginID(), $obj);
		}

		$obj = new \stdClass();
		$obj->cAktiv = 'Y';
		$obj->nFehlerhaft = '0';
		$this->getDB()->update('temailvorlage', 'kPlugin', $plugin->getID(), $obj);

		$minify = new MinifyService();
		$minify->flushCache();

		Shop::Container()->getCache()->flushTags(array(CACHING_GROUP_PLUGIN, CACHING_GROUP_TEMPLATE, 'cache_' . $plugin->getPluginID()));
		\error_log('updated from ' . $oldVersion . ' to ' . $newVersion);
	}


	public function uninstalled(bool $deleteData = true) {
		$plugin = $this->getPlugin();
		if ($deleteData === true) {
			$query = "SELECT * FROM INFORMATION_SCHEMA.TABLES WHERE table_schema = '" . DB_NAME . "' AND table_name like '" . $plugin->getPluginID() . "%'";
			$obj = $this->getDB()->query($query, 2);
			if ($obj !== null && count($obj)) {
				foreach ($obj as $oTable) {
					$this->getDB()->query('DROP TABLE IF EXISTS ' . $oTable->TABLE_NAME, 3);
				}
			}
		}
		$this->getDB()->delete('css_licence_log', 'cPluginID', $this->getPlugin()->getPluginID());
		Shop::Container()->getCache()->flushTags(array(CACHING_GROUP_PLUGIN, CACHING_GROUP_TEMPLATE, 'cache_' . $plugin->getPluginID()));

		$minify = new MinifyService();
		$minify->flushCache();

		parent::uninstalled($deleteData);
	}


	public function getPluginLink($oPlugin) {
		if (version_compare(APPLICATION_VERSION, "5.2.0-beta", ">=") === true) {
			return Shop::getAdminURL() . '/' . Route::PLUGIN . '/' . $oPlugin->kPlugin;
		} else {
			return Shop::getAdminURL() . '/' . "plugin.php?kPlugin=" . $oPlugin->kPlugin;
		}
	}


	public function addSubscriptionNotify(Notification $notification, $oUpdate) {
		$days = ceil(((int)$oUpdate->notifyTimeSubscription - time()) / 86400);
		if ((int)$days <= 0) return;
		$notification->add(
			NotificationEntry::TYPE_DANGER,
			(string)$oUpdate->cName,
			((int)$days == 1) ? "<div>Ihre Subscription f&uuml;r " . $oUpdate->cDomain . "</div>l&auml;uft morgen ab." : "<div>Ihre Subscription f&uuml;r " . $oUpdate->cDomain . "</div> l&auml;uft in  " . $days . " Tagen ab.",
			$this->getPluginLink($oUpdate),
			\md5($oUpdate->kPlugin . '_subscriptionInfo_' . mb_convert_encoding($oUpdate->notifyUpdate, 'ISO-8859-1', 'UTF-8'))
		);
	}


	public function addUpdateNotify(Notification $notification, $oUpdate) {
		$notification->add(
			NotificationEntry::TYPE_WARNING,
			(string)mb_convert_encoding($oUpdate->cName, 'UTF-8', mb_list_encodings()),
			"Update verf&uuml;gbar Version " . mb_convert_encoding($oUpdate->notifyUpdate, 'UTF-8', mb_list_encodings()),
			$this->getPluginLink($oUpdate),
			\md5($oUpdate->kPlugin . '_updateInfo_' . $oUpdate->notifyUpdate)
		);
	}

	public function checkNotification(Notification $notification): void {
		if (Shop::has('cssCheckNotification')) return;
		Shop::set('cssCheckNotification', true);

		if ($this->columnExist('css_licence_log', 'notifySubscriptionMail') == false) return;

		$query = "
					SELECT
					    p.cName,
					    p.kPlugin,
					    li.*
					FROM
					    css_licence_log AS li
					LEFT JOIN tplugin AS p
					ON
					    MD5(p.cPluginID) = MD5(li.cPluginID)
					WHERE
					    (
					        notifyUpdate != '' OR notifyTimeSubscription > 0
					    ) AND p.nStatus = 2	
		";


		$aData = $this->getDB()->query($query, 2);


		if ($aData !== null && count($aData) > 0) {
			$aHasNotify = array();
			foreach ($aData as $oData) {
				if (!empty($oData->notifyTimeSubscription)) {
					$this->addSubscriptionNotify($notification, $oData);
				}
				if (!empty($oData->notifyUpdate) && !in_array($oData->kPlugin, $aHasNotify)) {
					$aHasNotify[] = $oData->kPlugin;
					$this->addUpdateNotify($notification, $oData);
				}
			}
		}
	}
}
